#include "UnitZealotManager.h"
#include "UnitZealotInterface.h"
#include "InformationManager.h"
#include "Config.h"


void UnitZealotManager::onFrame()
{
	if (!(BWAPI::Broodwar->getFrameCount() <= 0 ||
		BWAPI::Broodwar->getFrameCount() - last_frame > Config::UnitZealotManager::FrequencyFrame))
		return; 

	if (target_region)
		attack();
	else
		scout(); 
}

void UnitZealotManager::onUnitCreate(BWAPI::Unit u)
{
	if (u->getPlayer() != BWAPI::Broodwar->self() ||
		u->getType().isBuilding())
		return;

	UnitZealot uz = new UnitZealotInterface(u); 
	units.push_back(uz); 
}


void UnitZealotManager::onUnitDestory(BWAPI::Unit u)
{
	if (u->getPlayer() != BWAPI::Broodwar->self() ||
		u->getType().isBuilding())
		return;

	std::vector<UnitZealot>::iterator iter = getUnitIterator(u); 
	if (iter == units.end())
		return; 

	delete (*iter); 
	iter = units.erase(iter); 
}


void UnitZealotManager::onEnd()
{
	for (std::vector<UnitZealot>::iterator iter = units.begin();
		iter != units.end();
		)
	{
		delete (*iter); 
		iter = units.erase(iter); 
	}
}


void UnitZealotManager::attack()
{
	if (!target_region || units.empty())
		return; 

	BWAPI::PositionOrUnit target = getTarget(); 
	BWAPI::Position target_position(-1, -1); 
	if (target.isUnit() && target.getUnit())
		target_position = target.getUnit()->getPosition();
	else if (target.isPosition())
		target_position = target.getPosition(); 

	if (!target_position.isValid())
		return; 
	
	std::pair<double, double> enemy_weapon_hp = getTargetEnemyWeaponAndHp(target_position);
	std::pair<double, double> self_weapon_hp = getSelfWeaponAndHp(target_position);
	double enemy_weapon = enemy_weapon_hp.first, enemy_hp = enemy_weapon_hp.second; 
	double self_weapon = self_weapon_hp.first, self_hp = self_weapon_hp.second;

	BWAPI::Unit cannon_unit = BWAPI::Broodwar->getClosestUnit(target_position,
		BWAPI::Filter::IsOwned && BWAPI::Filter::IsBuilding && BWAPI::Filter::CanAttack && BWAPI::Filter::IsPowered); 

	// can attack
	if ((self_hp / enemy_weapon) >= 6.0*(enemy_hp / self_weapon) ||
		(flag_attack_retreat && (self_hp / enemy_weapon) >= 2.0*(enemy_hp / self_weapon)))
		flag_attack_retreat = true;
	else
		flag_attack_retreat = false; 

	if (flag_attack_retreat ||
		(cannon_unit && 
		cannon_unit->getDistance(target) < BWAPI::UnitTypes::Protoss_Photon_Cannon.groundWeapon().maxRange()))
	{
		if (target.isUnit())
		{
			int attack_limit = 20; 
			switch (target.getUnit()->getType().size())
			{
			case BWAPI::UnitSizeTypes::Enum::Small:
				attack_limit = Config::UnitZealotManager::SmallSizeAttackLimit; 
				break; 
			case BWAPI::UnitSizeTypes::Enum::Medium:
				attack_limit = Config::UnitZealotManager::MediumSizeAttackLimit; 
				break; 
			case BWAPI::UnitSizeTypes::Enum::Large:
				attack_limit = Config::UnitZealotManager::LargeSizeAttackLimit;
				break;
			case BWAPI::UnitSizeTypes::Enum::Independent:
			case BWAPI::UnitSizeTypes::Enum::Unknown:
			case BWAPI::UnitSizeTypes::Enum::None:
				attack_limit = Config::UnitZealotManager::UnknownSizeAttackLimit;
				break;
			}
			for (auto& u : units)
			{
				if (!u->getUnit()->isCompleted())
					continue;
				if (attack_limit >= 0)
				{
					u->smartAttack(target.getUnit());
					attack_limit--;
				}
				else
					u->smartAttackMove(target.getUnit()->getPosition());
			}
		}
		else
		{
			for (auto& u : units)
			{
				if (!u->getUnit()->isCompleted())
					continue;
				u->smartMove(target.getPosition());
			}
		}
	}
	// just retreat
	else 
	{
		BWAPI::Position retreat_position(-1, -1); 
		if (cannon_unit)
			retreat_position = cannon_unit->getPosition(); 
		else 
			retreat_position = InformationManager::getInstance().getSelfRegion()->getCenter();

		UnitZealot closest_uz = getClosestUnit(target_position); 
		for (auto& u : units)
		{
			if (!u->getUnit()->isCompleted())
				continue; 
			if (u == closest_uz)
			{
				u->smartMove(retreat_position);
				continue;
			}
			int dist = u->getUnit()->getDistance(closest_uz->getUnit());
			if (dist < Config::UnitZealotManager::GatherRange)
				u->smartMove(retreat_position);
			else
				u->smartMove(closest_uz->getUnit()->getPosition());
		}
	}
}


void UnitZealotManager::scout()
{
	if (units.empty())
		return; 

	std::pair<Role, BWAPI::Position> scout_pair = InformationManager::getInstance().getDesireScoutTarget(units[0]->getUnit()->getPosition()); 
	if (scout_pair.second.isValid())
	{
		for (auto& u : units)
		{
			if (!u->getUnit()->isCompleted())
				continue;
			u->smartMove(scout_pair.second);
		}
	}
}


std::vector<UnitZealot>::iterator UnitZealotManager::getUnitIterator(BWAPI::Unit u)
{
	std::vector<UnitZealot>::iterator iter = units.begin();
	for (; iter != units.end(); iter++)
	{
		if ((*iter)->getUnit() == u)
			return iter; 
	}
	return iter; 
}


BWAPI::PositionOrUnit UnitZealotManager::getTarget()
{
	static BWAPI::PositionOrUnit last_target = nullptr; 

	if (!target_region)
		return BWAPI::PositionOrUnit(nullptr); 

	if (last_target.isUnit() && last_target.getUnit() && last_target.getUnit()->exists())
	{
		BWAPI::UnitType ut = last_target.getUnit()->getType(); 
		if ((ut.isBuilding() && ut.groundWeapon() != BWAPI::WeaponTypes::None) || ut == BWAPI::UnitTypes::Terran_Bunker)
			return last_target; 
	}

	InformationManager& IM = InformationManager::getInstance(); 

	std::list<BWTA::Chokepoint*> cp_path = IM.getRegionRegionPath(target_region, IM.getSelfRegion());
	BWAPI::Position source;
	if (cp_path.empty())
		source = IM.getSelfBaseLocation()->getPosition();
	else
		source = cp_path.front()->getCenter();

	double closest_dist_harm = 1e9, closest_dist_peace = 1e9; 
	BWAPI::Position harm_target_position(-1, -1), peace_target_position(-1, -1);
	BWAPI::Unit harm_target = nullptr, peace_target = nullptr; 
	for (auto& eb : IM.getRegionEnemyBuildings(target_region))
	{
		double dist = BWAPI::Position(eb.location).getDistance(source);
		if ((eb.unit_type.groundWeapon() != BWAPI::WeaponTypes::None || eb.unit_type == BWAPI::UnitTypes::Terran_Bunker) && 
			dist < closest_dist_harm)
		{
			closest_dist_harm = dist;
			harm_target_position = BWAPI::Position(eb.location);
			harm_target = BWAPI::Broodwar->getUnit(eb.id); 
			if (harm_target)
			{
				bool is = harm_target->exists();
				BWAPI::Position p = harm_target->getPosition();
			}
		}
		else if (eb.unit_type.groundWeapon() == BWAPI::WeaponTypes::None && dist < closest_dist_peace)
		{
			closest_dist_peace = dist; 
			peace_target_position = BWAPI::Position(eb.location); 
			peace_target = BWAPI::Broodwar->getUnit(eb.id); 
			if (peace_target)
			{
				bool is = peace_target->exists();
				BWAPI::Position p = peace_target->getPosition();
			}
		}
	}
	for (auto& eu : IM.getRegionEnemyUnits(target_region))
	{
		if (eu->isFlying())
			continue; 
		if (eu->getType() == BWAPI::UnitTypes::Zerg_Larva)
			continue; 
		if (eu->isMorphing() && !eu->getType().isBuilding())
			continue;

		BWAPI::WeaponType wt = eu->getType().groundWeapon(); 
		if (eu->isGatheringMinerals() || eu->isGatheringGas())
			wt = BWAPI::WeaponTypes::None; 

		double dist = eu->getDistance(source);
		if (wt != BWAPI::WeaponTypes::None && dist < closest_dist_harm)
		{
			closest_dist_harm = dist;
			harm_target_position = eu->getPosition();
			harm_target = eu;
		}
		else if (wt == BWAPI::WeaponTypes::None && dist < closest_dist_peace)
		{
			closest_dist_peace = dist; 
			peace_target_position = eu->getPosition(); 
			peace_target = eu; 
		}
	}

	int harm_target_wr = 32 * 7; 
	if (harm_target && harm_target->exists())
		harm_target_wr = harm_target->getType().groundWeapon().maxRange(); 
	if (harm_target_position.isValid() && 
		closest_dist_harm < closest_dist_peace + harm_target_wr + 32 * 4)
	{
		if (harm_target && harm_target->exists())
		{
			last_target = BWAPI::PositionOrUnit(harm_target);
			return BWAPI::PositionOrUnit(harm_target);
		}
		else
		{
			last_target = BWAPI::PositionOrUnit(harm_target_position);
			return BWAPI::PositionOrUnit(harm_target_position);
		}
	}
	else
	{
		if (peace_target && peace_target->exists())
		{
			last_target = BWAPI::PositionOrUnit(peace_target);
			return BWAPI::PositionOrUnit(peace_target);
		}
		else
		{
			last_target = BWAPI::PositionOrUnit(peace_target_position);
			return BWAPI::PositionOrUnit(peace_target_position);
		}
	}
}

std::pair<double, double> UnitZealotManager::getSelfWeaponAndHp(BWAPI::Position p)
{
	UnitZealot self_closest = getClosestUnit(p); 
	if (!self_closest)
		return std::pair<double, double>(1e-9, 1e-9); 

	std::vector<UnitZealot> radius_uzs = getUnitsInRadius(BWAPI::PositionOrUnit(self_closest->getUnit()), 
		Config::UnitZealotManager::WeaponSumRange); 
	radius_uzs.push_back(self_closest); 

	BWAPI::WeaponType wt = BWAPI::UnitTypes::Protoss_Zealot.groundWeapon(); 
	double weapon = 1e-9; 
	weapon += 2.0*double(wt.damageAmount()) * wt.damageFactor() / wt.damageCooldown() *
		radius_uzs.size(); 

	double hp = 1e-9; 
	for (auto& u : radius_uzs)
		hp += u->getUnit()->getHitPoints() + u->getUnit()->getShields(); 

	return std::pair<double, double>(weapon, hp); 
}


std::pair<double, double> UnitZealotManager::getTargetEnemyWeaponAndHp(BWAPI::Position p)
{
	InformationManager& IM = InformationManager::getInstance(); 

	double weapon = 1e-9, hp = 1e-9; 
	for (auto& u : BWAPI::Broodwar->getUnitsInRadius(p, 32 * 8, BWAPI::Filter::IsEnemy && BWAPI::Filter::CanAttack))
	{
		if (u->getType().groundWeapon() == BWAPI::WeaponTypes::None && 
			u->getType() != BWAPI::UnitTypes::Terran_Bunker)
			continue; 

		BWAPI::WeaponType wt = u->getType().groundWeapon();
		if (u->getType() == BWAPI::UnitTypes::Terran_Bunker)
		{
			wt = BWAPI::UnitTypes::Terran_Marine.groundWeapon(); 
		}
		
		if (u->getDistance(p) > wt.maxRange() + Config::UnitZealotManager::EnemyCoverRange)
			continue; 

		if (u->getType() == BWAPI::UnitTypes::Terran_Bunker)
		{
			weapon += 4.0*double(wt.damageAmount()) * wt.damageFactor() / wt.damageCooldown(); 
		}
		else if (u->getType() == BWAPI::UnitTypes::Protoss_Zealot)
			weapon += 2.0*double(wt.damageAmount()) * wt.damageFactor() / wt.damageCooldown();
		else
			weapon += double(wt.damageAmount()) * wt.damageFactor() / wt.damageCooldown();

		hp += u->getHitPoints() + u->getShields();
	}

	for (auto& eb : IM.getRegionEnemyBuildings(IM.getRegion(p)))
	{
		if (BWAPI::Broodwar->isVisible(eb.location))
			continue; 

		if (eb.unit_type.groundWeapon() == BWAPI::WeaponTypes::None &&
			eb.unit_type != BWAPI::UnitTypes::Terran_Bunker)
			continue;

		BWAPI::WeaponType wt = eb.unit_type.groundWeapon();
		if (eb.unit_type == BWAPI::UnitTypes::Terran_Bunker)
		{
			wt = BWAPI::UnitTypes::Terran_Marine.groundWeapon();
		}

		int dist = BWAPI::Position(eb.location).getDistance(p);
		if (dist > wt.maxRange() + Config::UnitZealotManager::EnemyCoverRange)
			continue;

		if (eb.unit_type == BWAPI::UnitTypes::Terran_Bunker)
		{
			weapon += 4.0*double(wt.damageAmount()) * wt.damageFactor() / wt.damageCooldown();
		}
		else
			weapon += double(wt.damageAmount()) * wt.damageFactor() / wt.damageCooldown();

		hp += eb.unit_type.maxHitPoints() + eb.unit_type.maxShields();
	}

	return std::pair<double, double>(weapon, hp);
}


UnitZealot UnitZealotManager::getClosestUnit(BWAPI::Position p)
{
	double closest_dist = 1e9; 
	UnitZealot closest_uz = nullptr; 
	for (auto& u : units)
	{
		if (!u->getUnit()->isCompleted())
			continue; 
		double dist = u->getUnit()->getDistance(p); 
		if (dist < closest_dist)
		{
			closest_dist = dist;  
			closest_uz = u;
		}
	}
	return closest_uz; 
}


std::vector<UnitZealot> UnitZealotManager::getUnitsInRadius(BWAPI::PositionOrUnit pu, int radius)
{
	std::vector<UnitZealot> radius_uzs; 
	for (auto& u : units)
	{
		if (!u->getUnit()->isCompleted())
			continue;
		double dist = 1e9;
		if (pu.isUnit())
			dist = u->getUnit()->getDistance(pu.getUnit());
		else if (pu.isPosition())
			dist = u->getUnit()->getDistance(pu.getPosition()); 
		if (dist < radius)
			radius_uzs.push_back(u); 
	}

	return radius_uzs; 
}



